// Copyright (c) The Bitcoin Core developers
// Distributed under the MIT software license, see the accompanying
// file COPYING or http://www.opensource.org/licenses/mit-license.php.

#include <bench/bench.h>
#include <cluster_linearize.h>
#include <test/util/cluster_linearize.h>
#include <util/bitset.h>
#include <util/strencodings.h>

#include <algorithm>
#include <cassert>
#include <cstdint>
#include <vector>

using namespace cluster_linearize;
using namespace util::hex_literals;

namespace {

/** Construct a wide graph (one root, with N-1 children that are otherwise unrelated, with
 *  increasing feerates). These graphs are pessimal for the LIMO step in Linearize, because
 *  rechunking is needed after every candidate (the last transaction gets picked every time).
 */
template<typename SetType>
DepGraph<SetType> MakeWideGraph(DepGraphIndex ntx)
{
    DepGraph<SetType> depgraph;
    for (DepGraphIndex i = 0; i < ntx; ++i) {
        depgraph.AddTransaction({int32_t(i) + 1, 1});
        if (i > 0) depgraph.AddDependencies(SetType::Singleton(0), i);
    }
    return depgraph;
}

template<typename SetType>
void BenchPostLinearizeWorstCase(DepGraphIndex ntx, benchmark::Bench& bench)
{
    DepGraph<SetType> depgraph = MakeWideGraph<SetType>(ntx);
    std::vector<DepGraphIndex> lin(ntx);
    bench.run([&] {
        for (DepGraphIndex i = 0; i < ntx; ++i) lin[i] = i;
        PostLinearize(depgraph, lin);
    });
}

void BenchLinearizeOptimallyTotal(benchmark::Bench& bench, const std::string& name, const std::vector<std::vector<uint8_t>>& serializeds)
{
    for (const auto& serialized : serializeds) {
        SpanReader reader{serialized};
        DepGraph<BitSet<64>> depgraph;
        reader >> Using<DepGraphFormatter>(depgraph);
        auto bench_name = strprintf("%s_%utx_%udep", name, depgraph.TxCount(), depgraph.CountDependencies());

        // Benchmark the total time to optimal.
        uint64_t rng_seed = 0;
        bench.name(bench_name).run([&] {
            auto [_lin, optimal, _cost] = Linearize(depgraph, /*max_iterations=*/10000000, rng_seed++);
            assert(optimal);
        });
    }
}

void BenchLinearizeOptimallyPerCost(benchmark::Bench& bench, const std::string& name, const std::vector<std::vector<uint8_t>>& serializeds)
{
    for (const auto& serialized : serializeds) {
        SpanReader reader{serialized};
        DepGraph<BitSet<64>> depgraph;
        reader >> Using<DepGraphFormatter>(depgraph);
        auto bench_name = strprintf("%s_%utx_%udep", name, depgraph.TxCount(), depgraph.CountDependencies());

        // Determine the cost of 100 rng_seeds.
        uint64_t total_cost = 0;
        for (uint64_t iter = 0; iter < 100; ++iter) {
            auto [_lin, optimal, cost] = Linearize(depgraph, /*max_iterations=*/10000000, /*rng_seed=*/iter);
            total_cost += cost;
        }

        // Benchmark the time per cost.
        bench.name(bench_name).unit("cost").batch(total_cost).run([&] {
            uint64_t recompute_cost = 0;
            for (uint64_t iter = 0; iter < 100; ++iter) {
                auto [_lin, optimal, cost] = Linearize(depgraph, /*max_iterations=*/10000000, /*rng_seed=*/iter);
                assert(optimal);
                recompute_cost += cost;
            }
            assert(total_cost == recompute_cost);
        });
    }
}

} // namespace

static void PostLinearize16TxWorstCase(benchmark::Bench& bench) { BenchPostLinearizeWorstCase<BitSet<16>>(16, bench); }
static void PostLinearize32TxWorstCase(benchmark::Bench& bench) { BenchPostLinearizeWorstCase<BitSet<32>>(32, bench); }
static void PostLinearize48TxWorstCase(benchmark::Bench& bench) { BenchPostLinearizeWorstCase<BitSet<48>>(48, bench); }
static void PostLinearize64TxWorstCase(benchmark::Bench& bench) { BenchPostLinearizeWorstCase<BitSet<64>>(64, bench); }
static void PostLinearize75TxWorstCase(benchmark::Bench& bench) { BenchPostLinearizeWorstCase<BitSet<75>>(75, bench); }
static void PostLinearize99TxWorstCase(benchmark::Bench& bench) { BenchPostLinearizeWorstCase<BitSet<99>>(99, bench); }

// Constructed from replayed historical mempool activity, selecting for clusters that are slow
// to linearize from scratch, with increasing number of transactions (9 to 63).
static const std::vector<std::vector<uint8_t>> CLUSTERS_HISTORICAL = {
    "8540e93a008331c66a018331c66a028331c66a038331c66a048331c66a058331c66a068f1b84a1060000000000000007995e888c04010000000000000000"_hex_v_u8,
    "8e4886a46000835c819a100000835d819c000100835f819c000200837581a7200300835f819c000400835c819a1005008360819c000600836e81a3400700835f819c0008009c728d813000000000000000000000854e88df0600008e48a9c5700100835f8bfd40000183608bfd40010100"_hex_v_u8,
    "836880bf4000834c80b6600000836880bf400100834c80b6600001836880bf400101834c80b6600002836880bf400102834c80b6600003836880bf400103834c80b6600004836880bf400104834c80b6600005836880bf400105836880bf400006834c80b6600106836880bf400106834c80b6600206836880bf400107834c80b6600207823c80bf400108834c80b660020800"_hex_v_u8,
    "81b0708db03800ef7a84a6700180c57b87c87802f24f83fc7e0381a3688ce20a04cb0d82a27205b40481887e068331965207d84384a07a0880850887b31609d57a83b62a09098865c82e0707c84083a736060009932d8098240a00058c1edf48070503f55a85fc560803058b6ddf660e0002e67d858d68090107921a81a1560d0302857cb32206148235986406149a5680e67c0b000106854cc12001118578e228001281cf6499b350120000000000108578e22801128578e228001300"_hex_v_u8,
    "80df2a87e05c00f62883d34801f23983c02c02808e0784ca280380c35b86d64c0480956584f1080580a91b85d20c06808c4b84c27c0780b473868c4408f54983cf7c0980aa5385d9380a80b00285f4140b80804f8487100c8099308582700d80b3138683640efe1183fa640fa734849e28000000000000000000000000000000000fc82788ce48010000000000000000000000000000000fa92b84be300200000000000000000000000000000000ab3b84e1460300000000000000000000000000000000833dc93c0102833dc93c02028331c14402038540e12603038431cc3405028335c50e030a8331c7760506930880af020307833dc07605068331be0205068335be3a05078d09fb76030d842dcb6e030d00"_hex_v_u8,
    "8368903600836890360183689036028368903603836890360483689036058368903606836890360783689036088368903609836890360a836890360b836890360c836890361b836890360e836890360f83689036228368903611836890361283689036138359900a13158359900a13158359900a13158359900a13158359900a13158359900a13158359900a13158359900a13158359900a13158359900a13158359900a13158359900a13158359900a12148359900a12148359900a11138359900a11138359900a11138359900a1113a42281a91012000000000000000000000000000000000000002600"_hex_v_u8,
    "83689220008734a05001836892200283689220038368922004854e99400583689220068368922007836892200406836892200306836892200207854e994009000983689220030c854e9940080107836892200309854e99400b0101836892200416854e9940070209854e994005060c854e9940020c0e854e9940030804854e9940030706891aa770070207854e9940040413854e994006020d854e99400311836892200310836892200217836892200113854e994003070d854e9940020410854e994002020d8734a05006010b854e9940050612854e994002128734a0500014854e9940050015836892200114854e994001090e854e99400507148734a050020b058734a0500210836892200222854e994004090e854e9940041200"_hex_v_u8,
    "db3284df6200e45386c550018235a906028235a906038236a906048235a906058600da0005058578d94c06058236a104070582429e10080582429b040905831d982e0a05854181e1040b058f1880f2240c058235a5040d058236a9060e05823581c3080f058235a90610058235a9061105857c80804012058235ab0613058235a90613058236a90614058236990215058474ca6e16058470ca6e17058a5480df0418058470ca6e19058316a43e1b04847c8088701b0586088086481d048236af061d058235a9061e058235a9061f058235b10620058236b10621058236b10622058236b1062305847080870224058546b34425058234b30626058235838f3227058242a504212a8241a9061b2d824181be76192d875282ad702b00168242ab061929845085af4c2d000f824181e448102b860480885c25168d0884c76e1b100000000100"_hex_v_u8,
    "8514fd1a00851480d02801903183803e028368e154038368e154048368e63c058368e63c068368e154078e4b82da4e08903181ec440983688187580a83688185240b83688185240c903181db100d8e4b8391180c0d8e4b84b4760c078361e566090b8361e5660b0a8363e1020d098363e1020f088363e1020c0c8e4b81be5614058e4b81dd6e0c0d835f8184200c0d835f8181700c0d835f8181700c0d8c6686a8100b24d63b93ea661205128e4b8391180d1d937d82d740071d8e4b81dd6e071c90318dce1604338e4b848d78042c8e4b82e234042c8d1e83f106052b8e4b81dd6e05258e4b85a9700333921785880c03328e4482e03c04308e4b8687380237920f85ab68020317178e4b86dd1a001c16d77ed4c56c020e0700000000068e4b8687380100318e4b868f1e00328e4b86ae3600328f2385f62a01181a836880fe100132845a81ab1e0033836881a93001338e438ffd30031c836881a9300135835f81a5400235835f81a5400137921790ba2c023697498dfd280800180019903189f81a003900"_hex_v_u8,
    "836280c06000836180c06001836280c06002836180c06003834280b1600303836180c0600403834280b1600403836180c0600503834280b1600503836280c0600603834280b1600603877e81fc20060c833a80b160060c836280c060070c877f81fc20070b834180b160070b877e81fc20070a836280c060070a834180b160080a880381fc200809836180c060081a834280b1600719836280c0600716836180c0600614834380b1600612836180c0600511834180b1600527834380b160041f834280b160041c836180c0600617836280c0600618834080b1600516877f81fc200531880281f960140414836280c060041e836280c060041f834280b160051f834580b160071e836280c060053b834280b1600422834880b1600423836180c0600622834280b1600343836180c0600443880281fc200f091d833e80b1600328834280b1600249836280c0600345877f81fc20140b23833b80b1600149877e81fc20020345877f81fc2011081f877e81fd400e0a12880281fb00110316877e81fc2003024a877f81fc20061b1c877f81fc200e0420877f81fc20040c1d877e81fc20021e20877f81fc20041d1f877e81fd40030e1c877f81fb00030a20877e81f9600c033200"_hex_v_u8
};

// Randomly generated clusters, selected for slow linearization, from 28 to 64 transactions.
// The first 10 are selected for long total runtime, the last 10 for high runtime per cost.
static const std::vector<std::vector<uint8_t>> CLUSTERS_SYNTHETIC = {
    "85058a7c008473b22d018706c70102873bd55303805e857b048732973005811487690686739032078260950508866b52098603a056000000000000000000000a8305986b0100000000000000000008852fa1260200000000000000000005855d8a0c03000000000000000000048468a9260400000000000000000004811f8449050000000000000000000483529f3e06000000000000000000048645962d07000000000000000000048333802a0800000000000000000001807187380900000000000000000001832f98220a000000000000000000018635a6380b00000000000000000001827c812c0c00000000000000000001844382700d00000000000000000000837d80410e00000000000000000000834ca16d0f00000000000000000000872d932c10000000000000000000008619b159110000000000000000000000"_hex_v_u8,
    "8040810400864384650184599d6902806c802903853fa502048332a25b058627cb6b068024897307861080790883469c7a098458b10f0a846bab240b8406a1770c8526b2290d870c710e840d8c630f8608926f108646af49118426a123128747c74213813b837d0000000000000000000000000000000000000000148629c81b010000000000000000000000000000000000000012865a857b020000000000000000000000000000000000000011866c8d3d03000000000000000000000000000000000000000f71871404000000000000000000000000000000000000000a86509a1805000000000000000000000000000000000000000a863e8c4b060000000000000000000000000000000000000009871db86a070000000000000000000000000000000000000006807b930a08000000000000000000000000000000000000000585779a2a090000000000000000000000000000000000000002831584430a00000000000000000000000000000000000000018156804a0b000000000000000000000000000000000000000100"_hex_v_u8,
    "8630852c008029804601864fb81d028602ae03038633af710481358b7a058559ad75068461a43b07865dd10a088516a42809810c8a250a821fa3150b812093470c805a85450d8708c27d0e8477aa770f8152850210806b892711810e805112821a904b13820582160000000000000000000000000000000000000000128114935f0100000000000000000000000000000000000000128179863902000000000000000000000000000000000000000e8064861103000000000000000000000000000000000000000d856a816c04000000000000000000000000000000000000000d8671b17405000000000000000000000000000000000000000d80368b0806000000000000000000000000000000000000000c847e881507000000000000000000000000000000000000000c8713811808000000000000000000000000000000000000000c801b89620900000000000000000000000000000000000000098139802b0a0000000000000000000000000000000000000009801d85440b00000000000000000000000000000000000000098161994d0c000000000000000000000000000000000000000882448c4b0d0000000000000000000000000000000000000008862b9c040e00000000000000000000000000000000000000068738e1580f000000000000000000000000000000000000000100"_hex_v_u8,
    "8345803600823fa12101862acf45028611882c038115953e0482748b51058606af0f068619a55f078436b76f086482440983358a380a846abb680b8164991f0c8142931b0d8535b1590e846d8d6d0f8400977b10811f8d6211857bc24f12857daf3300000000000000000000000000000000000000138262a8700100000000000000000000000000000000000011821d91240200000000000000000000000000000000000010870aa538030000000000000000000000000000000000001080788d68040000000000000000000000000000000000000b8240940e0500000000000000000000000000000000000009872f863c060000000000000000000000000000000000000781568460070000000000000000000000000000000000000282138e350800000000000000000000000000000000000002871c991609000000000000000000000000000000000000028072942a0a00000000000000000000000000000000000002835d0a0b00000000000000000000000000000000000002830b824f0c0000000000000000000000000000000000000282608e110d000000000000000000000000000000000000028071842f0e00000000000000000000000000000000000002837287410f00000000000000000000000000000000000001810e88671000000000000000000000000000000000000001867c977f11000000000000000000000000000000000000018704932612000000000000000000000000000000000000018724b61b13000000000000000000000000000000000000018339a072140000000000000000000000000000000000000100"_hex_v_u8,
    "8240952700865eb9710184219a7202863d8b2503833c9855048748b66705817ea16106814f8b1b07841f9349088717c90c09800d8d2d0a815190220b860bb4680c866d93250d81109a490e8075813e0f83388b3b1083268a7d118640c00f128727ae2813807b87781480428d5315833aaf6d168213a71d178623cc63188542a9531981796e1a855db8390000000000000000000000000000000000000000000000000000001b64883e0100000000000000000000000000000000000000000000000000001686638627020000000000000000000000000000000000000000000000000000168370871a03000000000000000000000000000000000000000000000000000015871b990104000000000000000000000000000000000000000000000000000014810e99080500000000000000000000000000000000000000000000000000000f80668e720600000000000000000000000000000000000000000000000000000f85738c240700000000000000000000000000000000000000000000000000000e803c8f660800000000000000000000000000000000000000000000000000000e873cb8200900000000000000000000000000000000000000000000000000000d8606cc3c0a00000000000000000000000000000000000000000000000000000b826f99050b000000000000000000000000000000000000000000000000000006815597400c000000000000000000000000000000000000000000000000000005860b955a0d00000000000000000000000000000000000000000000000000000581018f6b0e0000000000000000000000000000000000000000000000000000028644882e0f00000000000000000000000000000000000000000000000000000282048e0b1000000000000000000000000000000000000000000000000000000200"_hex_v_u8,
    "8041853f008115953c01866ea90e02816ba407038101870b04854eda0505841d995a06854ad1690783528b5e0886208a3709873de6580a8454af370b8732d21e0c846ecd410d822d9b190e8335b0440f80348137108278866d11874bd81d1284788d741385389e20148501cf0c15813b81251681108971000000000000000000000000000000000000000000000017813a9d7201000000000000000000000000000000000000000000001782448a6b020000000000000000000000000000000000000000000017855cb361030000000000000000000000000000000000000000000013833da46c04000000000000000000000000000000000000000000001281349c230500000000000000000000000000000000000000000000108367a120060000000000000000000000000000000000000000000010803b8b7c07000000000000000000000000000000000000000000000e8375864508000000000000000000000000000000000000000000000e874b901609000000000000000000000000000000000000000000000e8209942e0a000000000000000000000000000000000000000000000d83248e700b000000000000000000000000000000000000000000000d840d8b040c000000000000000000000000000000000000000000000d8619c2360d000000000000000000000000000000000000000000000d8027824f0e000000000000000000000000000000000000000000000d8135841e0f000000000000000000000000000000000000000000000d85349f0510000000000000000000000000000000000000000000000c8476a00611000000000000000000000000000000000000000000000b831caa4f12000000000000000000000000000000000000000000000b84318b7e13000000000000000000000000000000000000000000000b83319f3b1400000000000000000000000000000000000000000000048701b76515000000000000000000000000000000000000000000000481558c461600000000000000000000000000000000000000000000038529be4e170000000000000000000000000000000000000000000003835da02018000000000000000000000000000000000000000000000200"_hex_v_u8,
    "8639bf23008029863b0180539159028614c6330387408e22046782250582129117068365ad3a078075912708866e8a090982799c050a8474bf220b8548ba060c840eac020d850186460e815c85170f81628a301082369942118250a13b128644b74d13832da43314835c874f15800a8d5a00000000000000000000000000000000000000000000148707af630100000000000000000000000000000000000000000014806c993f020000000000000000000000000000000000000000001480288572030000000000000000000000000000000000000000001281449d7b04000000000000000000000000000000000000000000108505c536050000000000000000000000000000000000000000000f813e924b060000000000000000000000000000000000000000000f8510a24a070000000000000000000000000000000000000000000f8533875a080000000000000000000000000000000000000000000b8056800f090000000000000000000000000000000000000000000b870f8b0b0a0000000000000000000000000000000000000000000a87188b7a0b000000000000000000000000000000000000000000098417a87c0c00000000000000000000000000000000000000000009803d832c0d000000000000000000000000000000000000000000078663dd330e000000000000000000000000000000000000000000078511b21d0f000000000000000000000000000000000000000000058168992610000000000000000000000000000000000000000000058473bc4c110000000000000000000000000000000000000000000580428925120000000000000000000000000000000000000000000580408b61130000000000000000000000000000000000000000000581138d2814000000000000000000000000000000000000000000048647de091500000000000000000000000000000000000000000004817a85161600000000000000000000000000000000000000000004832d9d431700000000000000000000000000000000000000000003830b9512180000000000000000000000000000000000000000000382499e001900000000000000000000000000000000000000000003852d865e1a0000000000000000000000000000000000000000000380708f341b000000000000000000000000000000000000000000018555c1001c00000000000000000000000000000000000000000001822c886f1d0000000000000000000000000000000000000000000100"_hex_v_u8,
    "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"_hex_v_u8,
    "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"_hex_v_u8,
    "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"_hex_v_u8,
    "80578c04008734c974018654a51d02856aae0d038719973c0106807f904f04018430913d000a871a985d0500018374965f030109867466000e86479a690203038735c80d0109836997010700000003840ea121080000000082699354030310803a520702000483019f2709000000108159881f0703000e847d8f10060400128128901604070e851e8f1109020100801c820f01090009803d884a040101000000010f850e8f610202000106800b880606000200000100832e826f040100000000000009856fba6002020100000008865f9a0e06000000000000000100"_hex_v_u8,
    "81088137008624a1750180398f1602807a892b0102866bd12302028049450301803282010300076e880205000280068133050002810880170401088306997c05000002852e9e020401000282239424060001000166863504010000000d8135804d0601000000098227952105000100000a851a855e06000100000a8628a34b07000100000180038015070000010008801c8a7307000000010585168d210702000000000a821a6c03000202058329981f000200098507a21a000d8723bd13000417845cab5804020000000000000f83028f260201030380106e02010315842dac580501010012873bc37e040101001a8214973b060001000a80368d7902010100001900"_hex_v_u8,
    "820ea20c0083308407018237a67a02821fa37900048320ab4a0202844db6210401811b98490008830b931701068676ba5d030100058317997504000386259140020402857cc2410402000a8001832d010c81078d35060200068149960708010004863ea83505040d83498656070200038320a77503050111830b822f070200010681018041060302008232863b090200000085619c03030502088524a00c030002078545a04602050002028100921a0405000200008473c1530900000101000e845a8d080500020100000017867087460502010000000f7489310401010102018216a16a0104000001078427af750501000000010585458b43040202000002872cda4f0304010000068209851d010200000000138321827a01000000000f8356a908010000002200"_hex_v_u8,
    "833a8843008449995c01810a873000038066934d02018616c04d0202843a87600401826a864d0402855f8779050007863d936903038572d02d040204867cc61604030180098b69020509832792510601010581798e06040504833b9439070101000d851f946f01128147927d02030014866baa5d070201000c805f854f070300010b81579177090201000483209f7a0404020283678e420400050d8340981e080101000100048070816d07000101020c85578f010b01000000000004821a972605010204000010831c94280a01000000010001812d906c0303060882599e120505000200000a80638c5707020103844d9f04080004000000000e802b8c3107010300001b83189062020703000d872cb306080202000000098370a61e0600040000010d800a870f01010401000982369a1e030204000000058747c40c0102000317854e9e1c010300000000001a873eb04001020000002500"_hex_v_u8,
    "83778b4c00834c932901853ed03202834a844c00048716a22d02038529c429020103843ebe7f0300048135944b040002810f9704040005850b9463010980628e700501000a80238d380502078739843b080000008405896304000f80094c04010d807d9739070000000c8218a32304030109853a9d460500020f817b9412030600000987498636010014850e8104060400000b80378b5407030001000b8659d87f05040001000b831e8f740a000100000009844c9f7d040501188203964806040101038201946806050000188406bd48060600000014860e9c4f0b010000000f8549c114040500030f813b82510406000218842da63e0800030115811d9b5a0501010003001c84208029070004000d832a95220500010101010022852b9b24030401010000001a857f8107050104000000000f853e9525080100000000000109821f9507050101000104827aa04602030220834f803405000201001c8268aa1603030000000001008475b43202000300000005827b93560202000100002900"_hex_v_u8,
    "812d9c7200852fd301018170943702824a9e630387272402048353b33102038556b159020481599b00010883528735030381718370030107842cbf77050103816c8a1c0600048422c55d04078321a409090000000982688622050508806f8c090017873c5b08010004817fa04a050201048567c15f0505011082219f00080201118135892e07060000078360af6d0309138031824c0608001185739b080a020101000582018d0a0a000311813d860109010f8452bc6400060411825a8e2b050702098246a769012484049d3c07070100138313853a06080000158553b84b0b0203000000000010817da176030d1883388476040314852381580405050020830e966a0100042585128a3e08000001000121836f870403000700010000000025862cad000207020001128742d7180701010200010001088606da7c06000400010001000b84699b2d06030101000000010000138418b6250602030000001f84155c0401050013846a81610101030106873fd029020101000001000e82508521010300010d862ec12700000000001500"_hex_v_u8,
    "7d836f008521bb6b018061934b0285118b7903802285670203850f8b5201058221a327030287369342060080188c340601802d8f68060003840fb2400700028227a713070100028747c93d06010007874b99630503010281779f3103060380228e000602000f802f831a09000106857781300803000000108707a45d080001010011830a9132030811821aa471080301000010864aca7c0602021185758d640702000110843c8a2306000214850ec3040a01020100000082568231030a00000d810e927503000a0d8001803e070001021980209041010c0005822f9c6205070005835cbf50030b000011852bb20f0a0400000000001d83179f210c010101000000001a806a967c05030403208212980d0404030302832ca5660902020000001f852e9e39080303020000028572ca5208010203020013816192320e010000010000228672ef1d06060100011082419e6a050701021f8424b26e05040104010c8420bf780406030000000000855ac63c0b00010100000000000c8426ae25030501030016874a9330030104030019871be924060200010000000000118644cd2103010302010f8345ad03000401010d81379c2102000100010030837f810901010000000e832bb24f0100000600"_hex_v_u8,
    "816ba22900836f867101873c8056028511815803815497480205804283330105807e1d020383028a6100058533cc7b0100086a893d040003822493090302000b86108056020a854cbe4b05010483128c2d090000000182359815010c7281540800010002810889440602000e8518d60f09000000108563e05b0602020e8544ba220b000001018655da2105041382438a0c080202000381138a2b0702031085508f2904080004867ecb6709020002048375b142080103011a867bcc450508011983368a0b0c0200000a8420a84a0b03020000098312af7d0606020000088443c4550502050200068524b179090300000002001981188579030a00000217806d8f0a0c000005000d8076851a0800020003021581359369060101060103846eae34070203000200010022832a86670d0100000000000102830891620a00010100030000000001810592140900020001000003278712eb5d090500000100011f813f921a05020402128212ac1607010000090a870fc33001041981708e7a000903010a835ba272020900010001001884618f680207000101010118850bb168070301020000000680779a6f0406000002001f8670843605010500011e842e904709010003010000000d8222ae010205020128866086120702020000000000001082029c35010101030b82639c7802010300000100328138801001020102002000"_hex_v_u8,
    "815e9a23008324a10d01825bb246028470af1301048737ba3202048264aa000304843383590403857ba37c05028351be6805028123965a0700098567c53f0307854fde3e04000e825aa8790305098259b529020f8365985d0505068246ac510800010e83448c410215801b8464011c85548a540705010f840c9d10050800118472bb480211821aa7510d010000000010823b9d50010c108533a8030607020007804e90030d0000000a80799051060701078717900106070000168511962d0209000209806c801c060603001a8347a86607060102000f8245924309030001031a8177a513070504001b81348064080a0000000002856ce16106010902000f850088600701090200048044847b09040301010020842189140d04010000010000168210a91b02000609208618ae420609020000000b8257951b0b00070100000009855d8b7b0902030520810190300a0701000000000000028254ae7b0700060001031f8748bf21040902108264af760f0004000001000a813587660c0005000100000000002d8256a04a080a00000000246a8526030a06001780068011050a000403866ecf5a080a03000000008603c30509020000050f8250916201070306000025825385380e00000200000000348210836504090200010100048334b97a060502000201012c84088e520705020002010105814a8c65050507011a8178a15e05010101000000003680448c190200020300010100003280388f2c03000000010000000000002600"_hex_v_u8,
    "824da926008527804e01871bca36028604b04f038558d62c04804a960f02048513d87301068229897103058675a07e03000b843c8311050201814e5d02030c87029e020109841dc65a0601038227a325070101088334801c06030102853ee538070005688b0d080200010d8406a94d0b000000000b852bbf050a020000000a811380750d0000000005825c935d0b000200000a871bcb700902020109824187140f00000000098614ec7f0d000003078621942b080800058454bb1c0e030000000a847e630d030001000b8601e07e0c0002030c8655bf0b030f00018306bb5705010c1487038e071200000000001b856f816f1000000100020018820a9a530a0208000a81398a74001321853a944a060a03000117835aaf1c09030701118529b8690709020201248621bd330e010001010202148022800104100981428234090800030100188517cd2007060100020305836dc031020209022782449b69040c0100000001108651ac0a0607010101000000022a8337c0610505040202000001228547cd7c0d02000102020000000d843eba2a0b0501000000020025810f94010b0002030101000000010e803c921006000008041b8453bb3a0a0600030000000000000017841bae620708030100000016813f9924110100000200000000000000000c8637f36205060200050006857eb53f08020900000000002d810b9e580c0005000201000000000786459b700207060101188536b7790a0601000000010000000006854d905f08070201000000001680578f6a09030200030031837c8419080006000000022e8518a8500500040501000028861dd07e0801010300003b8743b97202000002000105003b8545b730010200010000000000001f00"_hex_v_u8
};

static void LinearizeOptimallyTotal(benchmark::Bench& bench)
{
    BenchLinearizeOptimallyTotal(bench, "LinearizeOptimallyHistoricalTotal", CLUSTERS_HISTORICAL);
    BenchLinearizeOptimallyTotal(bench, "LinearizeOptimallySyntheticTotal", CLUSTERS_SYNTHETIC);
}

static void LinearizeOptimallyPerCost(benchmark::Bench& bench)
{
    BenchLinearizeOptimallyPerCost(bench, "LinearizeOptimallyHistoricalPerCost", CLUSTERS_HISTORICAL);
    BenchLinearizeOptimallyPerCost(bench, "LinearizeOptimallySyntheticPerCost", CLUSTERS_SYNTHETIC);
}

BENCHMARK(PostLinearize16TxWorstCase, benchmark::PriorityLevel::HIGH);
BENCHMARK(PostLinearize32TxWorstCase, benchmark::PriorityLevel::HIGH);
BENCHMARK(PostLinearize48TxWorstCase, benchmark::PriorityLevel::HIGH);
BENCHMARK(PostLinearize64TxWorstCase, benchmark::PriorityLevel::HIGH);
BENCHMARK(PostLinearize75TxWorstCase, benchmark::PriorityLevel::HIGH);
BENCHMARK(PostLinearize99TxWorstCase, benchmark::PriorityLevel::HIGH);

BENCHMARK(LinearizeOptimallyTotal, benchmark::PriorityLevel::HIGH);
BENCHMARK(LinearizeOptimallyPerCost, benchmark::PriorityLevel::HIGH);
